import itertools
import random

from pkg_resources import safe_extra


class Minesweeper():
    """
    Minesweeper game representation
    """

    def __init__(self, height=8, width=8, mines=8):

        # Set initial width, height, and number of mines
        self.height = height
        self.width = width
        self.mines = set()

        # Initialize an empty field with no mines
        self.board = []
        for i in range(self.height):
            row = []
            for j in range(self.width):
                row.append(False)
            self.board.append(row)

        # Add mines randomly
        while len(self.mines) != mines:
            i = random.randrange(height)
            j = random.randrange(width)
            if not self.board[i][j]:
                self.mines.add((i, j))
                self.board[i][j] = True

        # At first, player has found no mines
        self.mines_found = set()

    def print(self):
        """
        Prints a text-based representation
        of where mines are located.
        """
        for i in range(self.height):
            print("--" * self.width + "-")
            for j in range(self.width):
                if self.board[i][j]:
                    print("|X", end="")
                else:
                    print("| ", end="")
            print("|")
        print("--" * self.width + "-")

    def is_mine(self, cell):
        i, j = cell
        return self.board[i][j]

    def nearby_mines(self, cell):
        """
        Returns the number of mines that are
        within one row and column of a given cell,
        not including the cell itself.
        """

        # Keep count of nearby mines
        count = 0

        # Loop over all cells within one row and column
        for i in range(cell[0] - 1, cell[0] + 2):
            for j in range(cell[1] - 1, cell[1] + 2):

                # Ignore the cell itself
                if (i, j) == cell:
                    continue

                # Update count if cell in bounds and is mine
                if 0 <= i < self.height and 0 <= j < self.width:
                    if self.board[i][j]:
                        count += 1

        return count

    def won(self):
        """
        Checks if all mines have been flagged.
        """
        return self.mines_found == self.mines


class Sentence():
    """
    Logical statement about a Minesweeper game
    A sentence consists of a set of board cells,
    and a count of the number of those cells which are mines.
    """

    def __init__(self, cells, count):
        self.cells = set(cells)
        self.count = count

    def __eq__(self, other):
        return self.cells == other.cells and self.count == other.count

    def __str__(self):
        return f"{self.cells} = {self.count}"

    def known_mines(self):
        """
        Returns the set of all cells in self.cells known to be mines.
        """

        # We can only know that a cell is a mine if all cells in a sentence are a mine
        if len(self.cells) == self.count and self.count > 0:
            return self.cells
        
        # else return empty set
        else:
            return set()


    def known_safes(self):
        """
        Returns the set of all cells in self.cells known to be safe.
        """

        # check if the length of the sentence cells is 0 and return
        if self.count == 0:
            return self.cells

        # else return empty set
        else:
            return set()



    def mark_mine(self, cell):
        """
        Updates internal knowledge representation given the fact that
        a cell is known to be a mine.
        """
        mine = cell
        
        # if mine is found in cells, remove and update count
        if mine in self.cells:
                # take the identified mine out of the sentence
                self.cells.remove(mine)
                # remove 1 count because mine was found and removed
                self.count -= 1

                

    def mark_safe(self, cell):
        """
        Updates internal knowledge representation given the fact that
        a cell is known to be safe.
        """
        safe = cell
        
        # if a safe cell is found, simply remove it
        if safe in self.cells:
            # take the identified safe out of the sentence
            self.cells.remove(safe)
            # don't update count because there's still a mine in there


class MinesweeperAI():
    """
    Minesweeper game player
    """

    def __init__(self, height=8, width=8):

        # Set initial height and width
        self.height = height
        self.width = width

        # Keep track of which cells have been clicked on
        self.moves_made = set()

        # Keep track of cells known to be safe or mines
        self.mines = set()
        self.safes = set()

        # List of sentences about the game known to be true
        self.knowledge = []

    def mark_mine(self, cell):
        """
        Marks a cell as a mine, and updates all knowledge
        to mark that cell as a mine as well.
        """
        self.mines.add(cell)
        for sentence in self.knowledge:
            sentence.mark_mine(cell)

    def mark_safe(self, cell):
        """
        Marks a cell as safe, and updates all knowledge
        to mark that cell as safe as well.
        """
        self.safes.add(cell)
        for sentence in self.knowledge:
            sentence.mark_safe(cell)

    def add_knowledge(self, cell, count):
        """
        Called when the Minesweeper board tells us, for a given
        safe cell, how many neighboring cells have mines in them.

        This function should:
            1) mark the cell as a move that has been made
            2) mark the cell as safe
            3) add a new sentence to the AI's knowledge base
               based on the value of `cell` and `count`
            4) mark any additional cells as safe or as mines
               if it can be concluded based on the AI's knowledge base
            5) add any new sentences to the AI's knowledge base
               if they can be inferred from existing knowledge
        """

        # 1) mark the cell as a move that has been made
        self.moves_made.add(cell)


        # 2) mark the cell as safe
        self.mark_safe(cell)


        # 3) add a new sentence to the AI's knowledge base based on the value of `cell` and `count`
        
        # 3.1) get all valid new neighborhood cells to add to a new sentence
        cell_i, cell_j = cell
        neighborhood = set()

        # get all cells in 9x9 grid around the cell
        for i in range ((cell_i - 1), (cell_i + 2)):
            for j in range ((cell_j - 1), (cell_j + 2)):
                # check to make sure that the cell is within the board
                if i in range(0, (self.height)) and j in range(0, (self.width)):

                    # check that cell isn't already in self.states (if it is simply continue)               
                    if (i, j) in self.safes:
                        continue
                    
                    # check if cell is already in mines & if so, subtract 1 from count and don't add cell to new sentence
                    elif (i, j) in self.mines:
                        count = count - 1
                        continue 

                    # check to make sure cell is not already in moves_made or the original cell
                    # if all requirements are fulfilled: add cell to new neighborhood of cells for new sentence
                    elif (i, j) not in self.moves_made and (i, j) != cell:
                        neighborhood.add((i, j))

        # 3.2) create new sentence with the count

        # instantiate new sentence with the above generated count and cells
        new_sentence = Sentence(neighborhood, count)
        
        # add to knowledgebase if there's actually anything in the sentence
        if len(new_sentence.cells) > 0:
            self.knowledge.append(new_sentence)



        # 4) mark any additional cells as safe or as mines if it can be concluded based on the AI's knowledge base
        
        # 4.1) if count == length of neighborhood -> set all neighbours as mines
        if count == len(neighborhood) and count > 0:
            for neighborhood_cell in neighborhood:
                self.mark_mine(neighborhood_cell)

        # 4.2) if count == 0 -> set all neighbors as safe
        elif count == 0:
            for neighborhood_cell in neighborhood:
                self.mark_safe(neighborhood_cell)



        # 5) add any new sentences to the AI's knowledge base if they can be inferred from existing knowledge

        cells_for_new_sentence = set()
        
        # check if the cells that make up the new sentence are a sub-part of any other sentence
        for sentence in self.knowledge:
            if new_sentence.cells in sentence.cells and len(sentence.cells) > len(new_sentence.cells):
                # get the count of the emcompassing sentence and calcuate the new count for the new sentence
                overall_count = sentence.count
                new_count = overall_count - new_sentence.count

                # add all other cells to a new sentence (i.e. the other sub-part), find this out by checking which cells are not in both sentences
                for cell in sentence.cells:
                    for new_cell in new_sentence.cells:
                        if cell != new_cell:
                            cells_for_new_sentence.add(cell)

                # create new sentence and add to knowledge base
                another_sentence = Sentence(cells_for_new_sentence, new_count)
                print(f"ADD KNOWLEDGE 5) Another sentence to be added to knowledge: {another_sentence}")
                self.knowledge.append(another_sentence)




    def make_safe_move(self):
        """
        Returns a safe cell to choose on the Minesweeper board.
        The move must be known to be safe, and not already a move
        that has been made.

        This function may use the knowledge in self.mines, self.safes
        and self.moves_made, but should not modify any of those values.
        """
        
        # initialise safe_move to None
        safe_move = None

        # iterate over self.safes and if there's one that hasn't already been played set it to safe_move & break
        for move in self.safes:
            if move not in self.moves_made:
                safe_move = move
                break
        
        return safe_move


    def make_random_move(self):
        """
        Returns a move to make on the Minesweeper board.
        Should choose randomly among cells that:
            1) have not already been chosen, and
            2) are not known to be mines
        """

        # initialise a random_move to None
        random_move = None

        # keep track of which random cells were already considered
        tried = set()
        
        # do until set c isn't 0 anymore (= valid random move found)
        c = 0
        while c == 0:
            # choose a random cell that is within the board
            random_cell = (random.randint(0, (self.height - 1)), random.randint(0, (self.width - 1)))

            # if new random cell is already tried, abandon this try
            if random_cell in tried:
                c = 0

            # check if all possible options have been considered -> if so, game is over and all mines are found
            if len(tried) == 64:
                c = 2
            
            else:
                # add random cell to tried
                tried.add(random_cell)

                # check if cell move has already been made
                if random_cell in self.moves_made:
                    c = 0
                
                # check if cell is in mines
                elif random_cell in self.mines:
                    c = 0
                
                # if made to here, conditions are checked
                else:
                    random_move = random_cell
                    # return False to end loop
                    c = 1

        # either return None if there is no valid move left or return the checked move
        return random_move